// script.js

// --- গেম কনফিগারেশন ---
const TOTAL_PUZZLES_PER_LEVEL = 5; // প্রতিটি লেভেলে 5টি পাজল থাকবে
const GRID_SIZE = 4; // 4x4 পাজল
const TOTAL_TILES = GRID_SIZE * GRID_SIZE; // 16
const TARGET_EMPTY_INDEX = 3; // ডানদিকের উপরের কোণা (0-ইনডেক্স 3)

// ⭐⭐ পিক্সেল-ভিত্তিক কনফিগারেশন সরানো হলো, কারণ শতাংশ ব্যবহার করা হবে ⭐⭐
// const BOARD_SIZE_PX = 500;
// const TILE_SIZE_PX = BOARD_SIZE_PX / GRID_SIZE; 


// লেভেলের ডেটা স্ট্রাকচার: আপনার ফাইল পাথ নিশ্চিত করুন!
const LEVEL_IMAGE_DIR = 'Level_Files'; 
const levelData = {
    1: Array.from({ length: TOTAL_PUZZLES_PER_LEVEL }, (_, i) => `${LEVEL_IMAGE_DIR}/Level_1/image_${i + 1}.jpg`),
    // 2: ... (আপনার অন্যান্য লেভেল)
};

// --- গেম স্টেট ভেরিয়েবল ---
let currentLevel = 1; 
let currentPuzzleIndex = 0; 
let tiles = []; 
let currentImageUrl = ''; 
let emptyTileIndex; 
let isGameActive = false;
let completedPuzzles = new Set(); // এই লেভেলে সম্পূর্ণ হওয়া পাজলগুলির ইনডেক্স ট্র্যাক করে

// --- DOM এলিমেন্ট ---
const levelSelectionEl = document.getElementById('level-selection'); 
const puzzlePathEl = document.getElementById('puzzle-path');
const puzzleAreaEl = document.getElementById('puzzle-area');
const puzzleBoardEl = document.getElementById('puzzle-board');
const nextPuzzleBtn = document.getElementById('next-puzzle-btn');
const winSectionEl = document.getElementById('win-section');

// --- ১. ইউটিলিটি ফাংশন ---

function shuffle(array) {
    for (let i = array.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [array[i], array[j]] = [array[j], array[i]];
    }
}

function isSolvable(arr) {
    let inversions = 0;
    for (let i = 0; i < arr.length; i++) {
        for (let j = i + 1; j < arr.length; j++) {
            if (arr[i] !== 0 && arr[j] !== 0 && arr[i] > arr[j]) {
                inversions++;
            }
        }
    }
    return inversions % 2 === 0;
}

function getPosition(index) {
    const row = Math.floor(index / GRID_SIZE);
    const col = index % GRID_SIZE;
    return { row, col };
}

// --- ২. লেভেল এবং পাজল ম্যানেজমেন্ট ---

function createLevelSelection() {
    puzzlePathEl.innerHTML = '';
    const totalPathItems = TOTAL_PUZZLES_PER_LEVEL; 
    let firstUncompletedIndex = -1;

    // প্রথম অসম্পূর্ণ পাজলটি খুঁজে বের করুন
    for(let i = 0; i < totalPathItems; i++){
        if(!completedPuzzles.has(i)){
            firstUncompletedIndex = i;
            break;
        }
    }

    for (let i = 0; i < totalPathItems; i++) {
        const pathItem = document.createElement('div');
        pathItem.classList.add('path-item');
        pathItem.textContent = `পাজল ${i + 1}`;
        
        if (completedPuzzles.has(i)) {
            pathItem.classList.add('unit-complete'); // সম্পূর্ণ নীল
        } else if (i === firstUncompletedIndex || i === currentPuzzleIndex) {
            // প্রথম অসম্পূর্ণ বা বর্তমানে নির্বাচিত পাজল খেলার যোগ্য
            pathItem.classList.add('playable'); // অসম্পূর্ণ সবুজ
            pathItem.dataset.puzzleIndex = i;
            pathItem.addEventListener('click', () => {
                currentPuzzleIndex = i;
                startPuzzle();
            });
        }
        
        puzzlePathEl.appendChild(pathItem);
    }
}

function startPuzzle() {
    // UI লুকানো
    levelSelectionEl.classList.add('hidden');
    puzzleAreaEl.classList.remove('hidden');
    winSectionEl.classList.add('hidden');
    
    loadPuzzle(currentLevel, currentPuzzleIndex); 
    isGameActive = true;
}

function showLevelSelectionScreen() {
    // UI দেখানো
    puzzleAreaEl.classList.add('hidden');
    levelSelectionEl.classList.remove('hidden');
    winSectionEl.classList.add('hidden');
    
    createLevelSelection(); 
}


function goToNextPuzzle() {
    isGameActive = false;
    
    // বর্তমান পাজলকে সম্পূর্ণ হিসাবে চিহ্নিত করুন
    completedPuzzles.add(currentPuzzleIndex); 

    // পরবর্তী অসম্পূর্ণ পাজল খুঁজে বের করুন
    let nextIndex = -1;
    for(let i = 0; i < TOTAL_PUZZLES_PER_LEVEL; i++){
        if(!completedPuzzles.has(i)){
            nextIndex = i;
            break;
        }
    }

    if (nextIndex !== -1) {
        currentPuzzleIndex = nextIndex;
        startPuzzle(); 
    } else {
        alert(`অভিনন্দন! লেভেল ${currentLevel} সম্পূর্ণ হয়েছে!`);
        // সব পাজল সম্পূর্ণ হলে, লেভেল সিলেকশন স্ক্রিনে ফিরে যান
        showLevelSelectionScreen(); 
    }
}

// --- ৩. পাজল গেমপ্লে লজিক ---

function loadPuzzle(level, index) {
    currentImageUrl = levelData[level][index]; 
    
    let tempTiles = Array.from({ length: TOTAL_TILES }, (_, i) => i + 1); 
    
    // TARGET_EMPTY_INDEX এর মানটিকে 0 দিয়ে প্রতিস্থাপন করা হলো (যেমন 4 -> 0)
    tempTiles[TARGET_EMPTY_INDEX] = 0; 
    
    do {
        shuffle(tempTiles);
    } while (!isSolvable(tempTiles)); 
    
    tiles = tempTiles;

    // 0 (খালি টাইল) কে TARGET_EMPTY_INDEX (3) এ সরানো হচ্ছে
    const emptyPosBeforeMove = tiles.indexOf(0);
    
    const tileAtTarget = tiles[TARGET_EMPTY_INDEX];
    tiles[TARGET_EMPTY_INDEX] = 0; 
    tiles[emptyPosBeforeMove] = tileAtTarget; 

    emptyTileIndex = TARGET_EMPTY_INDEX; 

    renderPuzzle();
}


function renderPuzzle() {
    puzzleBoardEl.innerHTML = '';
    puzzleBoardEl.style.gridTemplateColumns = `repeat(${GRID_SIZE}, 1fr)`;

    tiles.forEach((value, index) => {
        const tile = document.createElement('div');
        tile.classList.add('puzzle-piece');
        
        if (value === 0) { 
            tile.classList.add('empty-piece');
            emptyTileIndex = index;
        } else {
            const originalIndex = value - 1; 
            const { col: originalCol, row: originalRow } = getPosition(originalIndex);

            tile.style.backgroundImage = `url('${currentImageUrl}')`;
            // ব্যাকগ্রাউন্ড সাইজ 400% (4x4 গ্রিডের জন্য)
            tile.style.backgroundSize = `${GRID_SIZE * 100}% ${GRID_SIZE * 100}%`; 
            
            // ⭐⭐ পরিবর্তন: ব্যাকগ্রাউন্ড পজিশন শতাংশে সেট করুন ⭐⭐
            // x-অক্ষের পজিশন: (0/3 * 100)%, (1/3 * 100)%, (2/3 * 100)%, (3/3 * 100)%
            const percentX = (originalCol / (GRID_SIZE - 1)) * 100;
            const percentY = (originalRow / (GRID_SIZE - 1)) * 100;
            
            // শতাংশ পজিশনিং রেসপন্সিভ স্কেলিং নিশ্চিত করে
            tile.style.backgroundPosition = `${percentX}% ${percentY}%`; 

            tile.addEventListener('click', () => handleTileClick(index));
        }
        
        puzzleBoardEl.appendChild(tile);
    });
}

function handleTileClick(clickedIndex) {
    if (!isGameActive) return;
    
    const emptyPos = getPosition(emptyTileIndex);
    const clickedPos = getPosition(clickedIndex);

    const isAdjacent = 
        (Math.abs(emptyPos.row - clickedPos.row) === 1 && emptyPos.col === clickedPos.col) ||
        (Math.abs(emptyPos.col - clickedPos.col) === 1 && emptyPos.row === clickedPos.row);

    if (isAdjacent) {
        [tiles[clickedIndex], tiles[emptyTileIndex]] = [tiles[emptyTileIndex], tiles[clickedIndex]];
        
        emptyTileIndex = clickedIndex;
        renderPuzzle();
        checkWin();
    }
}

function checkWin() {
    let solved = true;
    const tempTiles = [...tiles]; 

    // 0 কে তার আসল মান (4) দিয়ে প্রতিস্থাপন করুন
    const indexOfZero = tempTiles.indexOf(0);
    if (indexOfZero !== -1) {
        tempTiles[indexOfZero] = TARGET_EMPTY_INDEX + 1; 
    }

    // সমাধানের লজিক
    for (let i = 0; i < TOTAL_TILES; i++) {
        if (tempTiles[i] !== i + 1) { 
            solved = false;
            break;
        }
    }

    if (solved) {
        // সমাধান: ডানদিকের উপরের কোণার খালি টাইলটিকে তার আসল ইমেজ পিস দিয়ে পূর্ণ করা
        
        const emptyTileInDOM = puzzleBoardEl.querySelector('.empty-piece');
        
        if (emptyTileInDOM) {
            emptyTileInDOM.classList.remove('empty-piece');
            emptyTileInDOM.style.backgroundImage = `url('${currentImageUrl}')`;
            emptyTileInDOM.style.backgroundSize = `${GRID_SIZE * 100}% ${GRID_SIZE * 100}%`;
            
            // 4 নম্বর পিসের আসল ব্যাকগ্রাউন্ড পজিশন সেট করুন (originalIndex 3)
            const originalIndex = TARGET_EMPTY_INDEX; 
            const { col: originalCol, row: originalRow } = getPosition(originalIndex);

            // ⭐⭐ পরিবর্তন: ব্যাকগ্রাউন্ড পজিশন শতাংশে সেট করুন ⭐⭐
            const percentX = (originalCol / (GRID_SIZE - 1)) * 100;
            const percentY = (originalRow / (GRID_SIZE - 1)) * 100;
            emptyTileInDOM.style.backgroundPosition = `${percentX}% ${percentY}%`;
        }
        
        puzzleCompleted();
    }
}

function puzzleCompleted() {
    isGameActive = false;
    winSectionEl.classList.remove('hidden'); 
}

// --- ৪. শুরু করা ---

// ইভেন্ট লিসেনার সেটআপ
nextPuzzleBtn.addEventListener('click', showLevelSelectionScreen); 

// যখন DOM লোড হবে
document.addEventListener('DOMContentLoaded', () => {
    // গেম শুরু হলে সবসময় লেভেল সিলেকশন স্ক্রিন দেখান
    showLevelSelectionScreen(); 
});